
-- PhysX global parameters

-- Plugin
global nvpx

-- physx panel parameters
global physXpaneldata = PhysXPanelInterface.instance

-- icon number
global PxIconCount = 15

-- global big values
global PxMaxValue =  100000000

-- Max Unit and Physics
-- It seems Max values are in unit of centimeter no matter what it displays. And SPEC asks to display all PhysX values in unit of meter.
global gPxVolumeUnitChange

-- if user's game is using a different geometry scale, user may wants to change the geometry scale in physics also.
global gPxGeometryScale = 1
global gPxUseGround = true
global gPxGroundHeight = 0.0
global gPxUseMultiThread = false
global gPxUseHardwareScene = false

-- SDK version related
global PxSDK3Loaded = false

-- define behavior modes
global gPxReasonPlayback = nvpxConsts.PX_HOST_PLAYBACK
global gPxReasonBake = nvpxConsts.PX_HOST_BAKE
global gPxReasonUnbake = nvpxConsts.PX_HOST_UNBAKE
global gPxReasonExport = nvpxConsts.PX_HOST_EXPORT
global gPxReasonSettle = nvpxConsts.PX_HOST_SETTLE   -- not supported
global gPxReasonModifierNone = 0
global gPxReasonModifierCancel = nvpxConsts.PX_HOST_MODIFIER_CANCEL
global gPxReasonModifierSelected = nvpxConsts.PX_HOST_MODIFIER_SELECTED

-- physx simulation states
global gPxSimulationDisabled = nvpxConsts.PX_SIM_DISABLED
global gPxSimulationPreparing = nvpxConsts.PX_SIM_PREPARING
global gPxSimulationRuning = nvpxConsts.PX_SIM_RUNNING
global gPxSimulationRewind = nvpxConsts.PX_SIM_REWINDING

-- global g_gravity
-- helpers
global nvConstraint
global nvSpring
global nvRagdoll
global PhysXModRB

-- alternative plugin name 
global ModRBClass = PhysXModRB
	
global PX_PHYSTYPE_RB_OVER    = nvpxConsts.PX_ACTORTYPE_RB_OVER --4     -- not used any more
global PX_PHYSTYPE_UNDEFINED  = nvpxConsts.PX_ACTORTYPE_UNDEFINED --5     
global PX_PHYSTYPE_CLOTH      = nvpxConsts.PX_ACTORTYPE_CLOTH --6
global PX_PHYSTYPE_SOFTBODY   = nvpxConsts.PX_ACTORTYPE_SOFTBODY --7
global PX_PHYSTYPE_FORCEFIELD = nvpxConsts.PX_ACTORTYPE_FORCEFIELD --8
global PX_PHYSTYPE_FLUID      = nvpxConsts.PX_ACTORTYPE_FLUID --9
global PX_PHYSTYPE_METAL_CLOTH  = nvpxConsts.PX_ACTORTYPE_METAL_CLOTH --10
global PX_PHYSTYPE_CLOTHING     = nvpxConsts.PX_ACTORTYPE_CLOTHING --14
global PX_PHYSTYPE_SBCLOTHING	= nvpxConsts.PX_ACTORTYPE_SBCLOTHING --17
global PX_PHYSTYPE_DESTRUCTION	= nvpxConsts.PX_ACTORTYPE_DESTRUCTION --18

-- PhysX Rigidbody Interactivity Types Definitions
global PX_PHYSTYPE_DYNAMIC    = nvpxConsts.PX_RBTYPE_DYNAMIC 
global PX_PHYSTYPE_KINEMATIC  = nvpxConsts.PX_RBTYPE_KINEMATIC
global PX_PHYSTYPE_STATIC     = nvpxConsts.PX_RBTYPE_STATIC

-- PhysX baking
global PX_PHYSBAKED_DYNAMIC = 2
global PX_PHYSBAKED_KINEMATIC = 3

-- PhysX RB mesh/collision Types
global PX_MESHTYPE_SPHERE	=	nvpxConsts.PX_SHAPE_SPHERE
global PX_MESHTYPE_BOX		=	nvpxConsts.PX_SHAPE_BOX
global PX_MESHTYPE_CAPSULE	=	nvpxConsts.PX_SHAPE_CAPSULE
global PX_MESHTYPE_CONVEX	=	nvpxConsts.PX_SHAPE_CONVEX
global PX_MESHTYPE_ORIGINAL =	nvpxConsts.PX_SHAPE_ORIGINAL
global PX_MESHTYPE_CUSTOM	=	nvpxConsts.PX_SHAPE_CUSTOM
global PX_MESHTYPE_COMPOSITE=	nvpxConsts.PX_SHAPE_COMPOSITE

-- physx layer: used to show node in wireframe mode
--global PxLayer
--global PxMaxWireMode
global PxColorDynamicRB  -- wireColor
global PxColorKinematicRB 
global PxColorStaticRB

-- init wire frame material
--if PxMaxWireMode == undefined then PxMaxWireMode = shell_material bakedMaterial:(standard wire:true) viewportMtlIndex:1
if PxColorDynamicRB == undefined then  (PxColorDynamicRB = color 10 90 140;)
if PxColorKinematicRB == undefined then (PxColorKinematicRB = color 230 150 0;)
if PxColorStaticRB == undefined then (PxColorStaticRB = color 200 200 200;)
-- colors 
global PxMatHullDynamicRB = shell_material bakedMaterial:(standard wire:true diffuse:PxColorDynamicRB) viewportMtlIndex:1
global PxMatHullKinematicRB = shell_material bakedMaterial:(standard wire:true diffuse:PxColorKinematicRB) viewportMtlIndex:1
global PxMatHullStaticRB = shell_material bakedMaterial:(standard wire:true diffuse:PxColorStaticRB) viewportMtlIndex:1

-- ragdoll List
global PxRagdollList = #()

fn PxSetRBHullNodeColor n interactivity =
(
	if n == undefined then false  -- result value
	else
	(
		n.material = case interactivity of 
		(
		PX_PHYSTYPE_DYNAMIC: (n.wirecolor = PxColorDynamicRB; PxMatHullDynamicRB)
		PX_PHYSTYPE_KINEMATIC: (n.wirecolor = PxColorKinematicRB; PxMatHullKinematicRB)
		PX_PHYSTYPE_STATIC: (n.wirecolor = PxColorStaticRB; PxMatHullStaticRB)
		)
	)
)

-- Define unit exchange rates
global RATE_RAD_TO_DEGREE = 180.0/pi
global RATE_DEGREE_TO_RAD = pi/180.0

-- main panels

-- simulation parameters
--global gPxClock
global gPxAnimationLoopOn = false
global gPxUseFirstFrame = true
global gPxFirstPhysXFrame = 0

global gPxOnLastFrame	= nvpxConsts.PX_ONLASTFRAME_CONTINUE	-- 1: Continue Sim, 2: Stop Sim, 3: Loop Anim
global gPxLoopAnimation	= nvpxConsts.PX_LOOP_RESET				-- 1: Reset Sim, 2: Continue Sim

-- version tag
global px_current_version = nvpxConsts.PX_PANELPROP_VERSION
global px_plugin_version
global px_version_converting = false

-- System unit related
global px_plugin_unittype = -1
global px_plugin_unitscale = 1.0

-- physics sdk parameters
global px_sdk_skinwidth
global px_sdk_contactDistance
global px_rb_solveritertions
global px_sdk_max_angvel = nvpxConsts.PX_SDK_MAXANGVEL

--global px_sdk_sleeplinevel
--global px_sdk_sleepangvel
global px_sdk_sleep_energy

global px_sdk_bouncethresh
global px_sdk_dynamicfrictionscaling
global px_sdk_staticfrictionscaling
global px_sdk_maxangvel
global px_sdk_ccd_epsilon
global px_sdk_sub_sim_steps
global px_sdk_continuous_cd
global px_sdk_ccd_motion_threshold

-- advanced setting radio buttons
global gPxSleepThresholdsAutomatic
global gPxCCDMinSpeedAutomatic
global gPxBounceMinSpeedAutomatic

-- US 6220
global gPxGenerateShapePerElement

-- Destruction
global gPxDestructionDamage
global gPxDestructionMomentum
global gPxDestructionRadius

-- physics scene parameters
global px_sdk_enable_gravity
global px_sdk_gravityDirection
global px_sdk_gravityx
global px_sdk_gravityy
global px_sdk_gravityz

-- current selected node
global pxCurrentNodeType = PX_PHYSTYPE_UNDEFINED
global pxCurrentNode = undefined
global pxCurrentShape = undefined
global pxSelectionChangedDirty = false
global pxSelectedNodes = #()
global pxSelectedRigidBodyMods = #()
global pxSelectedRigidBodyObjs = #()
global pxSelectedShapes = #()
global pxSelectedHelpers = #()
global pxSelectedDummyHelpers = #()
global pxSelectedConstraints = #()
global pxGroupNodes = #()
global pxSingleNodes = #()
global pxShapeHandles = #()
global px_UpdateSelectedRigidBodyLists
global px_selectionChanged

-- define ui related globals
global gPxRBPropertyPanel
global gPxD6PropertyPanel
global gPxClothPropertyPanel
global gPxSoftBodyPropertyPanel
global gPxFluidsPropertyPanel
global gPxEitterPropertyPanel


global gPxDialogWidth = 220
global gPxButtonWidth = 85
global gPxUIOffset = -5
global gPxIconOffset = 0

global gPxMaxDialogHight = 500
if nvpx != undefined then gPxMaxDialogHight = nvpx.getScreenHeight() - 200
--format "gPxMaxDialogHight = %\n" gPxMaxDialogHight

global gPxCreateConstraintTypeFlag = 0
global gPxCreateConstraintRigidMode = nvpxConsts.PX_CONSTRAINT_RIGID
global gPxCreateConstraintSlideMode = nvpxConsts.PX_CONSTRAINT_SLIDE
global gPxCreateConstraintHingeMode = nvpxConsts.PX_CONSTRAINT_HINGE
global gPxCreateConstraintTwistMode = nvpxConsts.PX_CONSTRAINT_TWIST
global gPxCreateConstraintUnivMode = nvpxConsts.PX_CONSTRAINT_UNIV
global gPxCreateConstraintBallSocketMode = nvpxConsts.PX_CONSTRAINT_BALLSOCK
global gPxCreateConstraintGearMode = nvpxConsts.PX_CONSTRAINT_GEAR

-- DOF/constraint mode
global PX_DOF_LOCKED = nvpxConsts.PX_DOF_LOCKED
global PX_DOF_LIMITED = nvpxConsts.PX_DOF_LIMITED
global PX_DOF_FREE	 = nvpxConsts.PX_DOF_FREE

-- Joint projection mode
global gPxJointProjLinear = 0
global gPxJointProjAngular= 1

fn PxGetIniFilename =
(
	(getdir #plugcfg) + "\\NvPhysXPlugin.ini"
)

-- global parameters save and load 
fn PxGetFileProperty pg name default forceDefault =
(
	if (forceDefault == true) then	default
	else
	(
		i = fileProperties.findProperty pg name
		if i == 0 then default else (fileProperties.getPropertyValue pg i)
	)
)

fn PxExistFileProperty pg name = 
(
	i = fileProperties.findProperty pg name
	if i == 0 then
		false
	else
		true
)
 	
fn PxUpdateGravity =
(
	if px_sdk_enable_gravity then (
		nvpx.setGravity [px_sdk_gravityx, px_sdk_gravityy, px_sdk_gravityz]
	) else (
		nvpx.setGravity [0, 0, 0]
	)
)

function PxUpdatePhysXParameters =
(
	if (nvpx == undefined) then 0  -- result value
	else
	(
		-- default visualizations
		nvpx.setSDKParameter "NX_SKIN_WIDTH" px_sdk_skinwidth
		nvpx.setSDKParameter "ContactShellOffset" px_sdk_contactDistance --physXpaneldata.contactShellContactDistance
		if nvpx.GetPhysXSDKVersionMajor() == 2 do
		(
			nvpx.setSDKParameter "NX_DYN_FRICT_SCALING" px_sdk_dynamicfrictionscaling
			nvpx.setSDKParameter "NX_STA_FRICT_SCALING" px_sdk_staticfrictionscaling
			nvpx.setSDKParameter "NX_CCD_EPSILON" px_sdk_ccd_epsilon
		)
		
		--deprecated sleep setting
		--nvpx.setSDKParameter "NX_DEFAULT_SLEEP_LIN_VEL_SQUARED" (px_sdk_sleeplinevel * px_sdk_sleeplinevel)
		--nvpx.setSDKParameter "NX_DEFAULT_SLEEP_ANG_VEL_SQUARED" (px_sdk_sleepangvel * px_sdk_sleepangvel * RATE_DEGREE_TO_RAD * RATE_DEGREE_TO_RAD)
		--nvpx.setSDKParameter "NX_DEFAULT_SLEEP_ENERGY" (0.005 * PxMeterToSystemUnit * PxMeterToSystemUnit)
		
		--mass invariant sleep energy
		nvpx.setSDKParameter "NX_DEFAULT_SLEEP_ENERGY" px_sdk_sleep_energy
		
		nvpx.setSDKParameter "NX_BOUNCE_THRESHOLD" px_sdk_bouncethresh
		nvpx.setSDKParameter "NX_MAX_ANGULAR_VELOCITY" px_sdk_maxangvel

		if px_sdk_continuous_cd == true then
			nvpx.setSDKParameter "NX_CONTINUOUS_CD" 1.0f
		else
			nvpx.setSDKParameter "NX_CONTINUOUS_CD" 0.0f
		nvpx.SetSimulationSubSteps px_sdk_sub_sim_steps
	
		/*
		nvpx.setSDKParameter "NX_VISUALIZATION_SCALE" 1.0
		nvpx.setSDKParameter "NX_VISUALIZE_JOINT_LOCAL_AXES" 1.0
		nvpx.setSDKParameter "NX_VISUALIZE_JOINT_WORLD_AXES" 1.0
		nvpx.setSDKParameter "NX_VISUALIZE_JOINT_LIMITS" 1.0
		nvpx.setSDKParameter "NX_VISUALIZE_WORLD_AXES" 1.0
		nvpx.setSDKParameter "NX_VISUALIZE_COLLISION_SHAPES" 1.0
		*/
	
		--format "DEBUG: update Physics SDK parameters ok.\n"
		1  -- result value
	)
)

fn PxUnitTypeToEnum unitType =
(
	case unitType of
	(
		#inches:		0
		#feet:			1
		#miles:			2
		#millimeters:	3
		#centimeters:	4
		#meters:		5
		#kilometers:	6
		default:		-1
	)
)

fn PxUnitEnumToType unitEnum = 
(
	case unitEnum of
	(
		0:		#inches
		1:		#feet
		2:		#miles
		3:		#millimeters
		4:		#centimeters 
		5:		#meters 
		6:		#kilometers
		default:
		(
				format "Unknown unit enum value: %!\n" unitEnum
			-1
		)
	)
)

-- show warning if trying to open a file created with later build
fn PxCheckFileVersion = 
(
	global physXpaneldata
	if physXpaneldata.pluginBuild != "" then
	(
		local t1 = filterstring physXpaneldata.pluginBuild "."
		local t2 = filterstring (nvpx.GetPluginBuild()) "."
		local isFileNewer = false
		if t1.count > 3 and t2.count > 3 do
		(
			for i = 1 to 3 do 
			(
				local v1 = t1[i] as integer
				local v2 = t2[i] as integer
				if v1 > v2 do (isFileNewer = true; exit;)
				if v1 < v2 do (isFileNewer = false; exit;)
				-- if equal, go on checking next parts
			)
		)
		if isFileNewer do messagebox (nvpxText.fileVersionCheckDlgWarning) title:(nvpxText.warningDlgTitle)
	)
)

fn PxShowPluginBuild =
(
	global physXpaneldata
	local msg = nvpxText.fileVersionIsEarlier
	if physXpaneldata.pluginBuild != "" then msg = physXpaneldata.pluginBuild
	msg = nvpxText.fileVersionDlgInfo + msg
	messagebox msg title:nvpxText.fileVersionDlgTitle
)

fn PxSaveGlobalParams =
(
	-- save plugin build version
	physXpaneldata.pluginBuild = (nvpx.GetPluginBuild())

	-- physics sdk part
	-- need update contact distance before skinWidth
	physXpaneldata.contactShellContactDistance = px_sdk_contactDistance
	physXpaneldata.skinWidth = px_sdk_skinwidth
)

fn PxGetSystemCommandMode = 
(
	sysCmdMode = toolMode.commandmode
		
	case sysCmdMode of
	(
		#SELECT:	0
		#MOVE:		1
		#ROTATE:	2
		#NUSCALE:	3
		#USCALE:	4
		#SQUASH:	5
		default:	-1
	)
)

fn PxSetSystemCommandMode sysCmdMode = 
(
	-- when setting the cmd mode, system accepts 6 modes only
	case sysCmdMode of
	(
		0:	toolMode.commandmode = #SELECT
		1:	toolMode.commandmode = #MOVE
		2:	toolMode.commandmode = #ROTATE
		3:	toolMode.commandmode = #NUSCALE
		4:	toolMode.commandmode = #USCALE
		5:	toolMode.commandmode = #SQUASH
	)
)

fn PxInitGlobalParamsForNewFile =
(
	-- for US4223 requirement B.1 B. Modification to Defaults
	-- multi thread and hardware scene
	gPxUseMultiThread = (if PHYSX_AUTODESK_VER != undefined then false else true)
	gPxUseHardwareScene = gPxUseMultiThread

	physXpaneldata.useMultiThread = gPxUseMultiThread
	physXpaneldata.useHardwareScene = gPxUseHardwareScene
)

fn PxSetContactDistanceFromRestDepth =
(
	physXpaneldata.contactShellContactDistance = px_sdk_skinwidth
	physXpaneldata.skinWidth = px_sdk_skinwidth
	px_sdk_contactDistance = physXpaneldata.contactShellContactDistance
)

fn PxLoadGlobalParam name &pbValue =
(
	i = fileProperties.findProperty #custom name
	if (i == 0) then
	(
		ret = pbValue	-- load from paramblock
	)
	else	-- for compatibility
	(
		ret = fileProperties.getPropertyValue #custom i	-- load from old file property
		pbValue = ret	-- assign value to physxpaneldata
	)
	ret
)

fn PxLoadFileProperties = 
(
	-- load system unit information
	px_plugin_unittype  = PxLoadGlobalParam "px_plugin_unittype" &physXpaneldata.unitType
	px_plugin_unitscale = PxLoadGlobalParam "px_plugin_unitscale" &physXpaneldata.unitScale

	if px_plugin_unittype == -1 then
	(
		px_plugin_unittype  = PxUnitTypeToEnum(units.SystemType)  	-- use current system settings
		px_plugin_unitscale = units.SystemScale
		physXpaneldata.unitType = px_plugin_unittype
	)

	-- let plugin know old system units	
	nvpx.ReadSystemUnit px_plugin_unittype  px_plugin_unitscale;

	-- read max unit information
	PxReadMaxUnit()
	
	-- version information 
	-- if global varible is not defined it is version 1.0
	px_plugin_version = PxLoadGlobalParam "px_plugin_version" &physXpaneldata.pluginPropertyVersion
	
	-- physics sdk part
	px_sdk_skinwidth = PxLoadGlobalParam "px_sdk_skinwidth" &physXpaneldata.skinWidth
	
	px_sdk_contactDistance = physXpaneldata.contactShellContactDistance
	px_rb_solveritertions = PxLoadGlobalParam "px_rb_solveritertions" &physXpaneldata.solverIteration
	px_rb_solveritertions = nvpx.setSolverIterations(px_rb_solveritertions);    -- update to plugin c++ part
	
	px_sdk_sleep_energy = PxLoadGlobalParam "px_sdk_sleep_energy" &physXpaneldata.sleepEnergy
	px_sdk_bouncethresh = PxLoadGlobalParam "px_sdk_bouncethresh" &physXpaneldata.bounceMinSpeedThreshold
	if px_sdk_bouncethresh > 0 then px_sdk_bouncethresh = -px_sdk_bouncethresh  --PhysX 2, the value ranges [-inf, 0)

	px_sdk_dynamicfrictionscaling = PxLoadGlobalParam "px_sdk_dynamicfrictionscaling" &physXpaneldata.dynamicFrictionScale
	px_sdk_staticfrictionscaling = PxLoadGlobalParam "px_sdk_staticfrictionscaling" &physXpaneldata.staticFrictionScale

	px_sdk_maxangvel = PxLoadGlobalParam "px_sdk_maxangvel" &physXpaneldata.maxAngleVelocity

	px_sdk_ccd_epsilon = PxLoadGlobalParam "px_sdk_ccd_epsilon" &physXpaneldata.ccdEpsilon

	-- US3547 - Substep Support, A.1.c, default to "0" for new projects. 
	px_sdk_sub_sim_steps = PxLoadGlobalParam "px_sdk_sub_sim_steps" &physXpaneldata.subSteps
	
	-- physics scene part
	px_sdk_enable_gravity = PxLoadGlobalParam "px_sdk_enable_gravity" &physXpaneldata.enableGravity
	px_sdk_gravityDirection = PxLoadGlobalParam "px_sdk_gravityDirection" &physXpaneldata.gravityDirection

	i = fileProperties.findProperty #custom "px_sdk_gravityx"
	oldGravityX = false
	if (i != 0) then
	(
		oldGravityX = true
		px_sdk_gravityx = fileProperties.getPropertyValue #custom i
	)
	
	i = fileProperties.findProperty #custom "px_sdk_gravityy"
	oldGravityY = false
	if (i != 0) then
	(
		oldGravityY = true
		px_sdk_gravityy = fileProperties.getPropertyValue #custom i
	)
	
	i = fileProperties.findProperty #custom "px_sdk_gravityz"
	oldGravityZ = false
	if (i != 0) then
	(
		oldGravityZ = true
		px_sdk_gravityz = fileProperties.getPropertyValue #custom i
	)
	
	if px_sdk_gravityDirection == 1 then
	(
		if oldGravityX == true then 
			physXPaneldata.gravity = px_sdk_gravityx 
		else 
			px_sdk_gravityx = physXPaneldata.gravity
	)

	if px_sdk_gravityDirection == 2 then
	(
		if oldGravityY == true then 
			physXPaneldata.gravity = px_sdk_gravityy 
		else 
			px_sdk_gravityy = physXPaneldata.gravity
	)

	if px_sdk_gravityDirection == 3 then
	(
		if oldGravityZ == true then 
			physXPaneldata.gravity = px_sdk_gravityz
		else 
			px_sdk_gravityz = physXPaneldata.gravity
	)

	--simulation part
	gPxOnLastFrame		= PxLoadGlobalParam "gPxOnLastFrame" &physXpaneldata.onLastFrame
	gPxLoopAnimation	= PxLoadGlobalParam "gPxLoopAnimation" &physXpaneldata.loopAnimation
		
	-- unit change part, always 1
	gPxGeometryScale = PxLoadGlobalParam "gPxGeometryScale" &physXpaneldata.geometryScale
	
	-- use ground
	gPxUseGround = PxLoadGlobalParam "gPxUseGround" &physXpaneldata.useGroundPlane
	gPxGroundHeight = PxLoadGlobalParam "gPxGroundHeight" &physXpaneldata.groundHeight
	
	-- multi thread and hardware scene
	gPxUseMultiThread = PxLoadGlobalParam "gPxUseMultiThread" &physXpaneldata.useMultiThread
	gPxUseHardwareScene = PxLoadGlobalParam "gPxUseHardwareScene" &physXpaneldata.useHardwareScene
	
	-- US 2946
	px_sdk_continuous_cd = PxLoadGlobalParam "px_sdk_continuous_cd" &physXpaneldata.enableCCD
	gPxSleepThresholdsAutomatic = PxLoadGlobalParam "gPxSleepThresholdsAutomatic" &physXpaneldata.sleepThresholdsAutomatic
	gPxCCDMinSpeedAutomatic = PxLoadGlobalParam "gPxCCDMinSpeedAutomatic" &physXpaneldata.ccdMinSpeedAutomatic
	gPxBounceMinSpeedAutomatic = PxLoadGlobalParam "gPxBounceMinSpeedAutomatic" &physXpaneldata.bounceMinSpeedAutomatic
	px_sdk_ccd_motion_threshold = PxLoadGlobalParam "px_sdk_ccd_motion_threshold" &physXpaneldata.ccdMinSpeedThreshold
	nvpx.SetCCDMotionThreshold px_sdk_ccd_motion_threshold
	
	gPxGenerateShapePerElement = PxLoadGlobalParam "gPxGenerateShapePerElement" &physXpaneldata.shapePerElement
	nvpx.SetShapePerElement gPxGenerateShapePerElement
	
	-- Destruction
	gPxDestructionDamage = PxLoadGlobalParam "gPxDestructionDamage" &physXpaneldata.destructionDamage
	gPxDestructionRadius = PxLoadGlobalParam "gPxDestructionRadius" &physXpaneldata.destructionRadius
	gPxDestructionMomentum = PxLoadGlobalParam "gPxDestructionMomentum" &physXpaneldata.destructionMomentum
	
	-- clear gPxCreationHelper
	gPxCreationHelper = undefined	
	
	-- check whether system units settings changes. see bug #5159: Gravity amount corrupted when scaling units on open
	-- for the case in bug #5159, there is not #unitsChange event. We have to check it manually
	-- when such a case happens, we only need to adjust some global parameters in PhysxToolPanel.
	-- Max will handle length related parameters. And we have some other codes to adjust RB hulls.
	oldMeterToSystemUnit = PxGetMeterToUnit(PxUnitEnumToType(px_plugin_unittype)) /  px_plugin_unitscale
	newMeterToSystemUnit = PxGetMeterToUnit(units.SystemType) /  units.SystemScale
	if (oldMeterToSystemUnit != newMeterToSystemUnit) then PxKeepGlobalParametersUnchanged(newMeterToSystemUnit/oldMeterToSystemUnit)

	-- US7664
	nvpx.useAdaptiveForce = physXpaneldata.useAdaptiveForce

	-- Gravity
	if (IsProperty nvpx "gravityobject") then
	(
		local propVal = PxLoadGlobalParam "px_gravityMode" &physXpaneldata.gravityMode
		if not px_sdk_enable_gravity then
			propVal = nvpxConsts.PX_GRAVITYMODE_NONE

		if propVal == 1 then
			nvpx.gravityMode = #none
		else if propVal == 2 then
			nvpx.gravityMode = #directional
		else if propVal == 3 then
			nvpx.gravityMode = #object

		-- gravity object
		i = fileProperties.findProperty #custom "px_gravityObject"
		if i != 0 then	-- check file property for compatibility
		(
			local gravityObjectName = fileProperties.getPropertyValue #custom i
			if (gravityObjectName != "") then
			(
				nvpx.gravityObject = Execute("$" + gravityObjectName)
				physXpaneldata.gravityObject = nvpx.gravityObject
			)
		)
		else
		(
			nvpx.gravityObject = physXpaneldata.gravityObject
		)
	)
)

fn PxResetGlobalParams = 
(
	-- load system unit information
	px_plugin_unittype  = PxUnitTypeToEnum(units.SystemType)
	px_plugin_unitscale = units.SystemScale
	
	-- let plugin know old system units	
	nvpx.ReadSystemUnit px_plugin_unittype  px_plugin_unitscale;

	-- read max unit information
	PxReadMaxUnit()

	-- version information 
	-- if global varible is not defined it is version 1.0
	px_plugin_version = nvpxConsts.PX_PANELPROP_VERSION

	-- physics sdk part
	px_sdk_skinwidth = (0.001 * PxMeterToSystemUnit)

	PxSetContactDistanceFromRestDepth()
	px_sdk_contactDistance = physXpaneldata.contactShellContactDistance
	
	px_rb_solveritertions = 30
	px_rb_solveritertions = nvpx.setSolverIterations(px_rb_solveritertions)

	px_sdk_sleep_energy = 0.05 * PxMeterToSystemUnit
	px_sdk_bouncethresh = (-2 * PxMeterToSystemUnit)
	px_sdk_dynamicfrictionscaling = 1
	px_sdk_staticfrictionscaling = 1 
	px_sdk_maxangvel = nvpxConsts.PX_SDK_MAXANGVEL
	px_sdk_ccd_epsilon = 0.01 * PxMeterToSystemUnit

	-- US3547 - Substep Support, A.1.c, default to "0" for new projects. 
	px_sdk_sub_sim_steps = 3
	
	-- physics scene part
	px_sdk_enable_gravity = true
	px_sdk_gravityDirection = 3		-- axis-Z

	px_sdk_gravityx = 0
	px_sdk_gravityy = 0
	px_sdk_gravityz = (-9.81 * PxMeterToSystemUnit)

	--simulation part
	gPxOnLastFrame		= nvpxConsts.PX_ONLASTFRAME_CONTINUE
	gPxLoopAnimation	= nvpxConsts.PX_LOOP_RESET
		
	-- unit change part
	gPxGeometryScale = 1
	
	-- use ground
	gPxUseGround = true
	gPxGroundHeight = 0.0

	-- multi thread and hardware scene
	gPxUseMultiThread = false 
	gPxUseHardwareScene = false

	-- US 2946
	px_sdk_continuous_cd = false
	gPxSleepThresholdsAutomatic = 1
	gPxCCDMinSpeedAutomatic = 1
	gPxBounceMinSpeedAutomatic = 1
	px_sdk_ccd_motion_threshold = 5
	nvpx.SetCCDMotionThreshold px_sdk_ccd_motion_threshold
	
	gPxGenerateShapePerElement = false
	nvpx.SetShapePerElement gPxGenerateShapePerElement
	
	-- Destruction
	gPxDestructionDamage = 1
	gPxDestructionRadius = 1
	gPxDestructionMomentum = 1 

	-- clear gPxCreationHelper
	gPxCreationHelper = undefined	
	
	oldMeterToSystemUnit = PxGetMeterToUnit(PxUnitEnumToType(px_plugin_unittype)) /  px_plugin_unitscale
	newMeterToSystemUnit = PxGetMeterToUnit(units.SystemType) /  units.SystemScale
	if (oldMeterToSystemUnit != newMeterToSystemUnit) then PxKeepGlobalParametersUnchanged(newMeterToSystemUnit/oldMeterToSystemUnit)

	-- US7664
	physXpaneldata.useAdaptiveForce = true
	nvpx.useAdaptiveForce = physXpaneldata.useAdaptiveForce

	-- Gravity
	if (IsProperty nvpx "gravityobject") then
	(
		local propVal = nvpxConsts.PX_GRAVITYMODE_DIR
		nvpx.gravityMode = #directional
		if not px_sdk_enable_gravity then
		(
			propVal = nvpxConsts.PX_GRAVITYMODE_NONE
			nvpx.gravityMode = #none
		)

		nvpx.gravityObject = undefined
		physXpaneldata.gravityObject = undefined
	)
)

fn PxUpdateGroundPlane =
(
	if gPxUseGround then (
		nvpx.CreateGround gPxGroundHeight [0, 0, 1]
	) else (
		nvpx.RemoveGround()
	)
)


fn PxSetGeometryScale scaleValue =
(
	nvpx.SetGeometryScale scaleValue
)

---
fn PxGetNodeByHandle handle =
(
	maxOps.getNodeByHandle handle
)

fn PxGetNodeHandle n =
(
	if n == undefined then 0 else n.inode.handle
)

fn PxGetNodeUserProp n key default = 
(
	if n == undefined then default
	else
	(
		ret = GetUserProp n key
		if undefined == ret then default 
		else ret
	)
)

filein "px_mouse.ms"
fn PxSimulateOneFrame =
(

	undo off
	(
		with animate off
		(
			local delta = 1.0/frameRate
		
			-- format "Simulating at % FPS or % Second\n" frameRate delta
			-- format "animation end is %\n" animationRange.end;

			if gPxAnimationLoopOn and (slidertime == animationRange.end) then slidertime = animationRange.start
			if(nvpx.GetAnimationState()) then
			(
				if(slidertime >= animationRange.end) then
				(
					slidertime = animationRange.end	-- continue simulation
				)
				else
					slidertime += 1;
			)
						
			-- if simulate continuously, gSimClock is enabled and SetBehaviorPlayMode as 1. 
			-- 1 is PlayModeEnum::CONTINUOUS. 0 is PlayModeEnum::STEP
			global gSimClock
			nvpx.SetBehaviorPlayMode (if gSimClock.Enabled then 1 else 0)
			nvpx.Simulate delta;
		)
	)
	if(false) then
	(
		local num = nvpx.findSleepingActors()
		for i = 1 to num do
		(
			n = nvpx.getSleepingActor (i-1)
			format " find sleeping actor: %\n" n
		)
	)
	if(false) then
	(
		-- print speed
		for n in $selection do
		(
			local vol = nvpx.getLinearVelocity n;
			format "velocity = % frameRate = %\n" vol frameRate;
		)
	)
	if(false) then
	(
		-- before using following command to read contacts, you need run command 'nvpx.SetUseContactReport true' before simulation to turn on contact-report
		-- and if you like, you can run 'nvpx.setContactFilter 30' before simulation to skip the contacts when force is below the filter.
		local num = nvpx.GetContactsCount();
		local index = 1;
		local hasContact = nvpx.SelectFirstContact();
		while(hasContact) do
		(
			local point = nvpx.GetContactPoint();
			local force = nvpx.GetContactForce();
			-- get the two actors in the collision
			local n1 = nvpx.GetContactNode0();
			local n2 = nvpx.GetContactNode1();
			format "contact % at point % force %, node % and %\n" index point force n1 n2;
			index = index + 1;
			hasContact = nvpx.GetNextContact();
		)
		-- other usages:
		for i = 1 to num do 
		(
			local point = nvpx.GetContactPoint index:i
			local force = nvpx.GetContactForce index:i
			-- get the two actors in the collision
			local n1 = nvpx.GetContactNode0 index:i
			local n2 = nvpx.GetContactNode1 index:i
			format "contact % at point % force %, node % and %\n" i point force n1 n2;
		)
	)

	/* TA5898, selection lost when you simulation. (by disable throwing codes)
	-- Deal with user dragging
	if nvpx.IsSimulating() == true then
	(
		g_dragger.onProceccing()
	)
	*/

)

fn PxGetModRBByClass node =
(
	local retval = undefined
	for i in node.modifiers while retval==undefined do
	(
		if classof(i) == ModRBClass do retval = i
	)	
	retval
)

fn PxGetModRB node =
(
	if node == undefined then undefined  -- result value
	else
	(	
		--format " PxGetModRB %\n" node
		modRB = PxGetModRBByClass(node)
		if isgrouphead(node) and (node.children.count > 0) do
			modRB = PxGetModRBByClass(node.children[1])
		modRB
	)
)

-- return false if it is not RB, true for RB
fn PxParseModRB node =
(
	modRB = PxGetModRB node
	if modRB == undefined then false  -- result value
	else
	(
		if (modRB.Density < 0.0000001) then modRB.Density = 0.0000001
		modRB.enabled  -- result value
	)
)

-- is bone or biped node
fn PxIsBone n =
(
	nodeclass = classof(n.baseobject)
	(n.boneEnable) or (nodeclass == BoneGeometry) or (nodeclass == Biped_Object) or (nvpx.IsBone n) or (nvpx.IsBiped n) or (PxGetNodeUserProp n "isBone" false)
)

-- get node position
fn PxGetNodePosition n =
(
	if classof(n) == Biped_Object then (
		biped.getTransform n #pos
	) else (
		n.transform.row4
	)
)

-- get node pivot
fn PxGetNodePivot n =
(
	if classof(n) == Biped_Object then (
		biped.getTransform n #pos  -- result value
	) else (
		n.transform.row4  -- result value
	)
)

global selectionChangeFn

fn PxGetSelectionCount =
(
	global pxSelectionChangedDirty
	global pxSelectedNodes
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedNodes.count
)

fn PxGetSelectedNodes =
(
	global pxSelectionChangedDirty
	global pxSelectedNodes
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedNodes
)

fn PxGetGroupNodesCount = 
(
	global pxSelectionChangedDirty
	global pxGroupNodes
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxGroupNodes.count
)

fn PxGetGroupNodes =
(
	global pxSelectionChangedDirty
	global pxGroupNodes
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxGroupNodes
)

fn PxGetSingleNodes =
(
	global pxSelectionChangedDirty
	global pxSingleNodes
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSingleNodes
)

fn PxGetSelectedHelpersCount = 
(
	global pxSelectionChangedDirty
	global pxSelectedHelpers
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedHelpers.count
)

fn PxGetSelectedHelpers =
(
	global pxSelectionChangedDirty
	global pxSelectedHelpers
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedHelpers
)

fn PxGetSelectedDummyHelpersCount = 
(
	global pxSelectionChangedDirty
	global pxSelectedDummyHelpers
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedDummyHelpers.count
)

fn PxGetSelectedDummyHelpers =
(
	global pxSelectionChangedDirty
	global pxSelectedDummyHelpers
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxSelectedDummyHelpers
)

fn PxGetCurrentNode =
(
	global pxSelectionChangedDirty
	global pxCurrentNode
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxCurrentNode
)

fn PxGetCurrentNodeType =
(
	global pxSelectionChangedDirty
	global pxCurrentNodeType
	global selectionChangeFn 
	if (pxSelectionChangedDirty == true) then
		selectionChangeFn()
	pxCurrentNodeType
)

-- hull generating algorithms
function PX_CreateBoundingBox n =
(
	t = nvpx.CreateBoundingBox n
	if t == undefined then undefined  -- result value
	else
	(
		t.name       = n.name + "_hull"
		t.pivot      = PxGetNodePivot(n)
		t  -- result value
	)
)

function PX_CreateBoundingSphere n =
(
	t = nvpx.CreateBoundingSphere n
	if t == undefined then undefined  -- result value
	else
	(
		t.name       = n.name + "_hull"
		t.pivot      = PxGetNodePivot(n)
		t  -- result value
	)
)


function PX_CreateConvexHullFromPoints n points maxVertices inflation=
(
	convexMesh = nvpx.createConvexFromPoints points maxVertices inflation
	if convexMesh == undefined then undefined  -- result value
	else
	(
		b.mesh = mesh mesh:convexMesh
		b.name = n.name + "_hull"
		b.transform = n.transform
		--b.pos  = PxGetNodePosition(n)
		b.pivot      = PxGetNodePivot(n)
		b  -- result value
	)
)

function PX_CreateBoundingCapsule n =
(
	t = nvpx.CreateBoundingCapsule n
	
	if t == undefined then undefined  -- result value
	else
	(
		t.name       = n.name + "_hull"
		t.pivot      = PxGetNodePivot(n)
		t  -- result value
	)
)

function PX_CreateBoundingConvex n =
(
	global PhysXModRB
	
	local maxVertices = 32
	local inflation   = 0
	modRB = PxGetModRB n
	if modRB != undefined do (maxVertices = modRB.meshVerticesLimit; inflation = modRB.meshInflation)

	convexMesh = nvpx.CreateBoundingConvex n maxVertices inflation
	if convexMesh == undefined then undefined  -- result value
	else
	(
		b.mesh = mesh mesh:convexMesh
		b.name = n.name + "_hull"
		b.pivot      = PxGetNodePivot(n)
		b  -- result value
	)
)

function PX_CreateBoundingConvexForNodes nodes =
(
	global PhysXModRB
	
	local maxVertices = 32
	local inflation   = 0
	firstNode = nodes[1]
	modRB = PxGetModRB firstNode
	if modRB != undefined do (maxVertices = modRB.meshVerticesLimit; inflation = modRB.meshInflation)

	convexMesh = nvpx.CreateBoundingConvexFromNodes nodes maxVertices inflation
	if convexMesh == undefined then undefined
	else
	(
		b.mesh = mesh mesh:convexMesh
		b.name = firstNode.name + "_hull"
		b.pivot      = PxGetNodePivot(firstNode)
		b
	)
)

fn PxIsClothingModifierSelected =
(
	if (PxGetSelectionCount() != 1) then false  -- result value
	else
	(
		selectedNodesArray = PxGetSelectedNodes()
		node = selectedNodesArray[1]
		if node == undefined then false  -- result value
		else
		(
			modClothing = node.modifiers["APEX Clothing"]
			modClothing != undefined  -- result value
		)
	)
)

fn PxIsRigidBody node = 
(
	(PxGetModRB node) != undefined
)

fn PxIsRB n =
(
	modRB = PxGetModRB n
	if modRB == undefined then false
	else
	(
		(modRB.type == PX_PHYSTYPE_DYNAMIC) or (modRB.type == PX_PHYSTYPE_KINEMATIC) or (modRB.type == PX_PHYSTYPE_STATIC)
	)
)

fn PxIsDynamicRB n =
(
	modRB = PxGetModRB n
	if modRB == undefined then false
	else (modRB.type == PX_PHYSTYPE_DYNAMIC)
)

fn PxIsKinematicRB n =
(
	modRB = PxGetModRB n
	if modRB == undefined then false
	else (modRB.type == PX_PHYSTYPE_KINEMATIC)
)

fn PxIsStaticRB n =
(
	modRB = PxGetModRB n
	if modRB == undefined then false
	else (modRB.type == PX_PHYSTYPE_STATIC)
)

fn PxIsPhysicsObject n =
(
	if PxIsRB n then true
	else
	(
		case (classof n) of
		(
			nvConstraint:		true
			--APEXClothing:		true
			default:			false
		)
	)
)

fn PxDeleteRB n =
(
	m = PxGetModRB n
	if m != undefined do (
		deleteModifier n m
	)
)


fn PxGetNodeType n =
(
	modRB = PxGetModRB n
	if modRB != undefined then modRB.type
	else (PxGetNodeUserProp n "PhysicsType" PX_PHYSTYPE_UNDEFINED)
)

/**********************************************
find the root parent node
**********************************************/
fn PxGetRootParent n =
(
	if n.parent == undefined then 
	(
		n
	)
	else
	(
		PxGetRootParent n.parent
	)
)

fn PxGetAllChildrenNodes n result =
(
	for i in n.children do (
		append result i
		PxGetAllChildrenNodes i result
	)
)

----
struct pxSelectionTool
(
	fn GetGroupRoot p =
	(
		if (isGroupMember p) then 
		(
			local pg = p;
			while (pg != undefined and (isGroupMember pg)) do
			(
				p = pg;
				pg = pg.parent;
			)
			if (isGroupHead pg) then (
				p = pg;
			)
		)
		p  -- result value
	),
	
	fn GetRootParent n =
	(
		if n.parent == undefined then 
		(
			n  -- result value
		)
		else
		(
			GetRootParent n.parent  -- result value
		)
	),
	
	/**********************************************
	find all the children nodes in a group 
	**********************************************/
	fn GetAllGroupChildren n result=
	(
		if n != undefined do
		(
			if (superclassof(n) == GeometryClass) do append result n
			if ((isGroupHead n) or (isGroupMember n)) do
			(
				if n.children.count > 0 do 
				(
					for i in n.children do
					(
						pxSelectionTool.GetAllGroupChildren i result;
					)
				)
			)
		)
	),
	
	fn parse nodes groups singles helpernodes=
	(
		if nodes == undefined then undefined  -- result value
		else
		(
			local firstNode = undefined
			if (classof(nodes)==ObjectSet) then
			(
				for n in nodes do
				(
					if(isGroupMember n) then
					(
						-- Autodesk wants to support helpers in group
						if superclassof(n) == helper then 
						(
							append helpernodes n
						)
						else 
						(
							if superclassof(n) == GeometryClass and (isOpenGroupMember n) then
							(
								append singles n
								if firstNode == undefined then firstNode = n
							)
						)
					)
					else
					(
						if(isGroupHead n) then 
						(
							append groups n
						)
						else
						(
							if (superclassof(n) != helper)  then  append singles n else append helpernodes n
						)
						if firstNode == undefined then firstNode = n
					)
				)
			)
			else
			(
				if(not isGroupMember nodes) then
				(
					if(isGroupHead nodes) then 
					(
						append groups nodes
					)
					else
					(
						-- we must ignore non-group helpers
						if superclassof(nodes) != helper then append singles nodes else append helpernodes nodes
					)
					if firstNode == undefined then firstNode = n
				)
			)
			firstNode  -- result value
		)
	)
)

pxSelectionTool = pxSelectionTool();

struct pxScaleTools
(
	fn isBoneOrBip n =
	(
		if n.boneEnable then true
		else ((classof n) == Biped_Object)
	),
	
	fn isBipedNode n =
	(
		local pos = undefined
		try(pos = biped.getTransform n #pos) catch()
		(pos != undefined)
	),

	/**********************************************
	Checks the node is scaled or not
	**********************************************/
	fn isScaled n =
	(
		-- assume it is biped node
		local retval = false
		try(
			--if isBipedNode n then (scale = biped.getTransform n #scale) else scale = n.scale
			scale = n.transform
			local escala = scale - [1,1,1]
			delta = dot escala escala
			retval = (delta > 0.001)
		)catch()
		retval  -- result value
	),

	fn ScaleBone n = 
	(
		parent = n.parent
		children = #()
		for i in n.children do
		(
			append children i
			i.parent = undefined
		)
		n.parent = undefined
		--format "!!! % is re-scaled!!!\n" n
		ResetXForm n
		for i in children do
		(
			i.parent = n
		)
		n.parent = parent
	),

	fn setNodePosition n value =
	(
		--if isBipedNode n then (	biped.setTransform n #pos value	) else (try(n.pos = value) catch()	)
		try( t = n.transform; t.row4 = value; n.transform = t) catch()
	),
	
	fn setNodePivot n value =
	(
		--if isBipedNode n then (	--biped.setTransform n #pos value)	else (try(n.pivot = value) catch()	)
		try(n.pivot = value) catch (setNodePosition n value)
	),
	
	fn getNodePosition n =
	(
		local value = undefined 
		--if isBipedNode n then (	value = biped.getTransform n #pos)	else ( try(value = n.pos) catch() )
		try( value = n.transform.row4 ) catch()
		value  -- result value
	),
	
	fn getNodePivot n =
	(
		local value = undefined 
		--if isBipedNode n then (	value = biped.getTransform n #pos)	else (try(value = n.pivot) catch() )
		try(value = n.pivot) catch(value = getNodePosition n)
		value  -- result value
	),
	
	fn alignPivotNodeAndProxy n proxy =
	(
		local pv = getNodePivot n
		local center = undefined
		--format "n = % proxy = %\n" n.name proxy.name
		try(center = n.center) catch(format "failed in pxScaleTools.alignPivotNodeAndProxy node = %, proxy = %." n.name proxy.name)
		--format "n pivot = % and center = %\n" pv center
		local t = center - pv
		local proxyCenter = proxy.center
		if (pv != undefined) then pxScaleTools.setNodePivot proxy (proxyCenter - t)
	),
	
	fn alignPosNodeAndProxy n proxy =
	(
		alignPivotNodeAndProxy n proxy
		local pos = getNodePosition n
		pxScaleTools.setNodePosition  proxy pos
	),
	
	fn CenterPivot n = 
	(
		try(
		  if n.pivot != n.center then n.pivot = n.center
		) catch()
	),
	
	-- for grouped nodes, they will be merged as one
	fn ScaleGeometryObject n =
	(
		if n.children.count > 0 then
		(
			for i in n.children do
			(
				if (superclassof(i) == GeometryClass) then i.parent = parent
			)
		)
		if n.parent != undefined then
		(
			if (superclassof(n.parent) == GeometryClass) then n.parent = undefined
		)
		--format "!!! % is re-scaled!!!\n" n
		ResetXForm n
	),
	
	fn ScaleGroupedObject n =
	(
		local children = #()
		local result = #()
		local missed = #()
		pxSelectionTool.GetAllGroupChildren n children
		-- format " there are % children: %\n" children.count children
		for i in children do
		(
			if (superclassof(i) == GeometryClass) and (isScaled i) then 
			(
				if i.children.count > 0 then
				(
					for k in i.children do
					(
						if (superclassof(k) == GeometryClass) then k.parent = n
					)
				)
				if i.parent != undefined then
				(
					if (superclassof(i.parent) == GeometryClass) then i.parent = n
				)
				-- format "!!! % is re-scaled!!!\n" i
				ResetXForm i
			)
		)
	),

	fn RemoveScales n =
	(
		if (isBipedNode n) then n  -- result value
		else if (superclassof(n) == GeometryClass) then
		(
			if (isScaled n) then 
			(
				if n.boneEnable then
				(
					ScaleBone n
				)
				else
				(
					ScaleGeometryObject n
				)
			)
		)
		else if(isGroupHead n) do
		(
			if(not isGroupMember n) do
			(
				ScaleGroupedObject n
			)
		)
		n  -- result value
	)
)

pxScaleTools = pxScaleTools();


--- add nodes to physics scene
fn findobject s =
(
	retval = undefined
	for o in objects while retval==undefined do
	(
		if (o.name == s) do retval = o 
	)
	retval  -- result value

)


fn px_add_rigidbody n =
(
	if not PxParseModRB(n) then false  -- result value
	else
	(
		ret = false
		try
		(
			PxSaveRBKeys(n) -- for determinism
			ret = ((nvpx.AddRigidbody n) == 1)
		) catch()
		ret  -- result value
	)
)

fn PxAddD6Joint n =
(
	--p = n.parent
	--if p == undefined then p = n.body0
	--if p != undefined then (
	--	modRB = PxGetModRB p
	--	if modRB != undefined then n.globalAxis = [modRB.InitialSpinX, modRB.InitialSpinY, modRB.InitialSpinZ]
	--	if n.globalAxis == [0, 0, 0] then n.globalAxis = [1, 0, 0]
	--)
	nvpx.AddConstraint n
)

fn PxAddSpring n =
(
	nvpx.AddDistanceJoint n
)

fn PxAddAllPhysXObjects nodes =
(
	nvpx.RemoveAll()

	target = #()

	groupNodes = #()
	singleNodes = #()
	helperNodes = #()
	dummyNodes = #()
	
	pxSelectionTool.parse nodes groupNodes singleNodes helperNodes
	for i in helperNodes do  if classof(i) == Dummy then  append dummyNodes i
	
-- 	format " singleNodes = %\n" singleNodes
-- 	format " groupNodes = %\n" groupNodes
-- 	format " helperNodes = %\n" helperNodes
	
	rigidbodies = #()
	
	--number = 0
	-- create ground first
	PxUpdateGroundPlane()
	
	-- other objects
	for grp in groupNodes do
	(
		for n in grp do
		(
			if classof(n) == Dummy then	
				continue()
			
			t = PxGetNodeType n
-- 			format " node % type = %\n" n.name t
			ok = case t of 
			(
				PX_PHYSTYPE_DYNAMIC: append rigidbodies n
				PX_PHYSTYPE_KINEMATIC: append rigidbodies n
				PX_PHYSTYPE_STATIC: append rigidbodies n
				default: false
			)
		)
	)

	for n in singleNodes do
	(
		t = PxGetNodeType n
		--format " node % type = %\n" n.name t
		ok = case t of 
		(
		PX_PHYSTYPE_DYNAMIC: append rigidbodies n
		PX_PHYSTYPE_KINEMATIC: append rigidbodies n
		PX_PHYSTYPE_STATIC: append rigidbodies n
		default: false
		)
	)	
	for n in dummyNodes do
	(
		t = PxGetNodeType n
		--format " node % type = %\n" n.name t
		ok = case t of 
		(
		PX_PHYSTYPE_DYNAMIC: append rigidbodies n
		PX_PHYSTYPE_KINEMATIC: append rigidbodies n
		PX_PHYSTYPE_STATIC: append rigidbodies n
		default: false
		)
	)
	
	nRB = 0
	for n in rigidbodies do
	(
		--format " try to add RB %\n" n
		if (px_add_rigidbody n) then nRB = nRB + 1
	)

	nJoint = 0
	for i in helperNodes do
	(
		if classof(i) == nvConstraint then
		(
			nJoint = nJoint + (PxAddD6Joint i)
		)
		else if classof(i) == nvSpring then
		(
			nJoint = nJoint + (PxAddSpring i)
		)
	)
	
	number = nRB + nJoint
	-- do not show the message
	--info = "PhysX Notice:";
	--if nRB > 0 then info = info + " " + (nRB as string) + " rigid bodies."
	--if nJoint > 0 then info = info + " " + (nJoint as string) + " constraints."
	--print info
	number  -- result value
)

fn PxForceCreatePhysXScene =
(
	nvpx.ConnectPVD "localhost"
	gPxFirstPhysXFrame = slidertime
	nvpx.BeforeAddPhysXActors()
	PxAddAllPhysXObjects objects; 
	nvpx.SetSimulationState(gPxSimulationPreparing)
	--format "creat all physx scene\n"
)

fn PxCaptureTransform useSelection =
(
	local targets = undefined
	
	if useSelection then
		targets = $selection
	else
		targets = $geometry
	
	if targets != undefined and targets.count > 0 then
		nvpx.CaptureInitTransforms targets useSelection
)

fn PxLoadSavedPose =
(
	local needRestore = PxLoadGlobalParam "px_restore_original_pose" &physXpaneldata.restoreOriginalPose

	if(	needRestore ) then
	(
		for i in objects do
		(
			modRB = PxGetModRB(i)
			if modRB != undefined then
			(
				i.transform = nvpx.GetInitialPose(i) --modRB.initialPose
			)
		)
	)
)

fn PxCreatePhysXScene =
(
	if (not nvpx.IsSimulating()) then 
	(
		PxForceCreatePhysXScene()
	)
	
)

fn PxResetPhysXScene =
(
	with redraw off 
	(
		-- CM: Reset the simulating bool before redrawing to stop extra redraws
		-- in visualizer
		
		nvpx.SimulationReset()
		PxResetRBKeys()
		PxClearSavedRBKeys()
	)
	redrawviews ();
)

-- for determinism support
fn PxUseMultiThread enable =
(
	nvpx.UseMultiThread enable
	PxResetPhysXScene()
)

fn PxUseHardwareScene enable =
(
	nvpx.UseHardwareScene enable
	PxResetPhysXScene()
)

-- US2946, set ccd threshold
fn PxSetCCDMotionThreshold ccdMotionThreshold = 
(
	nvpx.SetCCDMotionThreshold ccdMotionThreshold
	PxResetPhysXScene()
)

fn PxInitializePlugin = 
(
	try ( nvpx.InitializePhysX listener ) 
	catch ( format "PhysX Warning:  unable to call function nvpx.InitializePhysX from PhysX.dlm, the plugin might not be correctly installed."; )
	-- create toolbar
	nvpx.CreateToolbar()
)


fn PxIsBaked modRB =
(
	modRB.baked > 0
)

-- True if the object can be baked in general
fn PxIsBakeable modRB =
(
	(modRB.type == PX_PHYSTYPE_DYNAMIC) or (modRB.type == PX_PHYSTYPE_KINEMATIC and modRB.switchType)
)

-- True if the object can have a baked keyframe at the given time
fn PxIsBakeableAtTime modRB timeVal = 
(
	(modRB.type == PX_PHYSTYPE_DYNAMIC) or
	((modRB.type == PX_PHYSTYPE_KINEMATIC) and (modRB.switchType) and (timeVal >= modRB.switchTypeAtFrame))
)

fn PxMakeDynamicRB n =
(
	global PhysXModRB
	
	if isgroupmember(n) and not isopengroupmember(n) then undefined  -- result value
	else if isgrouphead(n) then 
	(
		format "Note: skip group node '%'\n" (n.name)
		undefined  -- result value
	)
	else
	(
		local modRB = PxGetModRB n
		if modRB == undefined then 
		(
			modRB = PhysXModRB()
			--format " modRB = %\n" modRB
			modRB.type = PX_PHYSTYPE_DYNAMIC
			if isgrouphead(n) then
			(
				children = #()
				pxSelectionTool.GetAllGroupChildren n children
				for i in children do addmodifier i modRB
			)
			else
			(
				addmodifier n modRB
			)
		)
		else
		(
			-- for baked, we can not modify its type
			if not (PxIsBaked modRB) do modRB.type = PX_PHYSTYPE_DYNAMIC
		)
	
		modRB  -- result value
	)
)


fn PxMakeKinematicRB n =
(
	global PhysXModRB
	if isgroupmember(n) then undefined  -- result value
	else if isgrouphead(n) then
	(
		format "Note: skip group node '%'\n" (n.name)
		undefined  -- result value
	)
	else
	(	
		local modRB = PxGetModRB n
		if modRB == undefined then 
		(
			modRB = PhysXModRB()
			modRB.type = PX_PHYSTYPE_KINEMATIC
			if isgrouphead(n) then
			(
				children = #()
				pxSelectionTool.GetAllGroupChildren n children
				for i in children do addmodifier i modRB
			)
			else
			(
				addmodifier n modRB
			)
		)
		else
		(
			-- for baked, we can not modify its type
			if not (PxIsBaked modRB) do modRB.type = PX_PHYSTYPE_KINEMATIC
		)
	
		modRB  -- result value
	)
)

fn PxMakeStaticRB n =
(
	global PhysXModRB
	if isgroupmember(n) then undefined  -- result value
	else if isgrouphead(n) then 
	(
		format "Note: skip group node '%'\n" (n.name)
		undefined  -- result value
	)
	else
	(
		local modRB = PxGetModRB n
		if modRB == undefined then 
		(
			modRB = PhysXModRB()
			modRB.type = PX_PHYSTYPE_STATIC
			modRB.meshType = PX_MESHTYPE_ORIGINAL
			if isgrouphead(n) then
			(
				subNodes = #()
				pxSelectionTool.GetAllGroupChildren n subNodes
				-- "subNodes=%\n" subNodes
				for i in subNodes do addmodifier i modRB
			)
			else
			(
				addmodifier n modRB
			)
		)
		else
		(
			-- for baked, we can not modify its type
			if not (PxIsBaked modRB) do modRB.type = PX_PHYSTYPE_STATIC
		)
	
		modRB  -- result value
	)
)

fn PxDynamicRBNodes nodes =
(
	for i in nodes do PxMakeDynamicRB i
)


fn PxKinematicRBNodes nodes =
(
	for i in nodes do PxMakeKinematicRB i
)

fn PxStaticRBNodes nodes =
(
	for i in nodes do PxMakeStaticRB i
)

-- MassFX Cloth Helpers

fn PxSeletionCanCreateCloth =
(
	-- in future we need check PxIsPhysicsObject()
	-- only if group node and geometry node selected, then we can make them as Cloth
	retval =  true
	for i in $selection while retval do
	(
		if not (IsGroupHead(i) or (superclassof(i) == GeometryClass)) do retval = false
	)
	
	retval  -- result value
)

fn PxGetModClothByClass node =
(
	local retval = undefined
	for i in node.modifiers while retval==undefined do
	(
		if classof(i) == mCloth do retval = i
	)	
	retval
)

fn PxGetModCloth node =
(
	if node == undefined then undefined  -- result value
	else
	(	
		modCloth = PxGetModClothByClass(node)
		if isgrouphead(node) and (node.children.count > 0) do
			modCloth = PxGetModClothByClass(node.children[1])
		modCloth
	)
)

fn PxMakeCloth n =
(
	global PhysXModRB
	
	if isgroupmember(n) and not isopengroupmember(n) then undefined  -- result value
	else if isgrouphead(n) then 
	(
		undefined  -- result value
	)
	else
	(
		local modCloth = PxGetModCloth n
		if modCloth == undefined then 
		(
			modCloth = mCloth()
			if isgrouphead(n) then
			(
				children = #()
				pxSelectionTool.GetAllGroupChildren n children
				for i in children do addmodifier i modCloth
			)
			else
			(
				addmodifier n modCloth
			)
		)
	
		modCloth  -- result value
	)
)

fn PxClothNodes nodes =
(
	for i in nodes do PxMakeCloth i
)


fn PxRemoveClothMod n =
(
	m = PxGetModCloth n
	while m != undefined do (
		deleteModifier n m
		m = PxGetModCloth n
	)
)

-- Tools Dialog Helpers

fn PxCloseMenu =
(
	if ((menuMan.findMenu "PhysX") != undefined) then 
	(
		menuMan.unRegisterMenu (menuMan.findMenu "PhysX")
		menuMan.updateMenuBar()
	)
)

fn PxCloseAllUI =
(
	--global px_utility
	
	--try( removerollout px_utility) catch()
	PxCloseMenu()
)

fn PxShowPhysicsPanel startPage =  -- TO DO: Migrate px_panel_methods
(
	global px_physXPanel
	global px_panel_globalParameters
	global px_panel_simulation

	if not px_physXPanel.open then
	(
		-- sync global parameters from script to c++
		PxSetPhysXPanelInterfaceFromGlobalParams()
		
		-- window pos
		position = px_physXPanel.LoadPos()
		pos = [position.x, position.y]
		wndSizeY = position.z
		width = 200

		-- Make this rollout resizeable
		fixedWndSizeRatio = 0.95
		wndSize = getMAXWindowSize()
		maxWndSizeY = (wndSize.y * fixedWndSizeRatio)
		if wndSizeY < 400 do wndSizeY = 400
		if wndSizeY > maxWndSizeY do wndSizeY = maxWndSizeY
		
		CreateDialog px_physXPanel width wndSizeY escapeEnable:false pos:pos
		cui.RegisterDialogBar px_physXPanel style:#(#cui_dock_left,#cui_dock_right,#cui_floatable,#cui_handles) minsize:[width,300] maxsize:[width,2000]

		--nvpx.PhysXPanelTooltip()
	)

	-- Always switch tabs, even if dialog is already open
	px_physXPanel.setCurrentPage startPage
	
	-- set focus to the dialog
	setFocus px_physXPanel
)


fn PxSimulateNFrames n =
(
-- 	progressstart "PhysX simulation"
	for i = 1 to n do 
	(
		with redraw off
		(
			PxSimulateOneFrame();
		)
		redrawviews ();
		i += 1;
-- 		progressupdate (100.0*i/n)
	)
-- 	progressend()
)

fn PxIsSimulationRunning =
(
	global gSimClock
	if gSimClock == false then undefined else gSimClock.Enabled
)

fn PxRunSimulation =
(
	global gPhysXMenuRunSimItem
	nvpx.ClearInvalidNodes()

	undo off
	(
		global gSimClock
		global gPxSimClockLastTime

		if (not nvpx.IsSimulating()) and gPxUseFirstFrame then slidertime = animationRange.start
		PxCreatePhysXScene()
	
		--px_utility.simClock.active = not px_utility.simClock.active
		--set gPxSimClockLastTime to prevent stuck in dead loop when burning time
		gPxSimClockLastTime = timeStamp() - 1000.0/(frameRate)		
		gSimClock.Enabled = not gSimClock.Enabled

		if not gSimClock.Enabled do nvpx.PauseSimulation()

		--updateToolbarButtons()
		nvpx.SetButtonCheck 1003 (PxIsSimulationRunning())
	)
)

fn PxPauseSimulation = 
(
	global gPhysXMenuRunSimItem
	undo off
	(
		if (not nvpx.IsSimulating()) then undefined  -- result value
		else
		(	
			global gSimClock
			gSimClock.Enabled = false

			--updateToolbarButtons()
			nvpx.SetButtonCheck 1003 (PxIsSimulationRunning())
		)
	)
)

fn PxStepSimulation =
(
	nvpx.ClearInvalidNodes()

	global gSimClock	
	gSimClock.Enabled = false
	
	if (not nvpx.IsSimulating()) and gPxUseFirstFrame then slidertime = animationRange.start
	if (sliderTime >= animationRange.end) then
	(	
		case gPxOnLastFrame of
		(
			1: slidertime = animationRange.end
			2: PxPauseSimulation()
			3: if gPxLoopAnimation == 1 then (PxStopSimulation()) else sliderTime = animationRange.start
		)
	)	
	PxCreatePhysXScene()
	PxSimulateNFrames 1

	--updateToolbarButtons()
	nvpx.SetButtonCheck 1003 (PxIsSimulationRunning())
)

fn PxStopSimulation =
(
	global gPhysXMenuRunSimItem
	if nvpx.GetSimulationState() != gPxSimulationDisabled then
	(
		undo off
		(
			--global px_utility
			global gSimClock
			gSimClock.Enabled = false
			--------------------------------
			-- CM: Bug 4207:  Only reset after cancelling simulation.
			-- CM: Bug 4287: reset simulating befor changing slider time and causing a paint.
			slidertime = animationRange.start
			PxResetPhysXScene()
			--updateToolbarButtons()
			nvpx.SetButtonCheck 1003 (PxIsSimulationRunning())
			nvpx.RemoveAll()
		)
	)
)

fn PxShowViewer =
(
	--format nvpxText.TXT_GLOBALS_SHOW_VIEWER
	
	PxStopSimulation()
	slidertime = animationRange.start
	
	nvpx.PreCreateViewer();
	
	gPxFirstPhysXFrame = slidertime
	PxAddAllPhysXObjects objects; 
	PxClearSavedRBKeys()
	
	nvpx.CreateViewer();

	-- remove all the physx objects from max, but keep the scene in the viewer
	PxStopSimulation()
	undo off
	(
		slidertime = animationRange.start
	)
	
	nvpx.PostCreateViewer();
	
)

fn PxSeletionCanCreateRB =
(
	-- in future we need check PxIsPhysicsObject()
	-- only if group node and geometry node selected, then we can make them as RB
	retval =  true
	for i in $selection while retval do
	(
		if not (IsGroupHead(i) or (superclassof(i) == GeometryClass) or (classof(i) == Dummy)) do retval = false
	)
	
	retval  -- result value
)


fn PxRefreshRBPropertyPanels =
(
	global px_rbPanel_basic
	global px_rbPanel_physicalMesh
	global px_rbPanel_material
	global px_rbPanel_display
	global px_rbPanel_Advanced
	
	currentNode =	PxGetCurrentNode()
	if currentNode != undefined then
	(
		if px_rbPanel_basic != undefined do (if px_rbPanel_basic.open then px_rbPanel_basic.loadValues() )
		if px_rbPanel_physicalMesh != undefined do (if px_rbPanel_physicalMesh.open then px_rbPanel_physicalMesh.loadValues() )
		if px_rbPanel_material != undefined do (if px_rbPanel_material.open then px_rbPanel_material.loadValues() )
		if px_rbPanel_display != undefined do (if px_rbPanel_display.open then px_rbPanel_display.loadValues() )
		if px_rbPanel_Advanced != undefined do (if px_rbPanel_Advanced.open then px_rbPanel_Advanced.loadValues() )
	)
)

fn PxRefreshD6PropertyPanels =
(
	global px_d6Panel_connection
	global px_d6Panel_translation
	global px_d6Panel_swingtwist
	global px_d6Panel_spring
	global px_d6Panel_advanced
	
	if PxGetSelectedHelpersCount() != 0 then
	(
		if px_d6Panel_connection != undefined do (if px_d6Panel_connection.open then px_d6Panel_connection.loadValues() )
		if px_d6Panel_translation != undefined do (if px_d6Panel_translation.open then px_d6Panel_translation.loadValues() )
		if px_d6Panel_swingtwist != undefined do (if px_d6Panel_swingtwist.open then px_d6Panel_swingtwist.loadValues() )
		if px_d6Panel_spring != undefined do (if px_d6Panel_spring.open then px_d6Panel_spring.loadValues() )
		if px_d6Panel_advanced != undefined do (if px_d6Panel_advanced.open then px_d6Panel_advanced.loadValues() )
	)
)

---------- ---------- ---------- ----------
-- RolloutInfo.  Handling to save state state of rollouts (expanded or collapsed) to ini file
---------- ---------- ---------- ----------

struct PxRolloutInfo_struct
(
	rolloutVal = undefined, -- The rollout
	rolloutRolledUp = undefined -- Whether to rollup is expanded or collapsed
)

-- Update the corresponding state information, for each currently visible rollout in the list
fn PxUpdateRolloutInfoList rolloutInfoList =
(
	for rolloutInfoVal in rolloutInfoList do
		if rolloutInfoVal.rolloutVal.isDisplayed then rolloutInfoVal.rolloutRolledUp = (not rolloutInfoVal.rolloutVal.open)
)

-- Load state info from ini file, for each rollout in the list
fn PxLoadRolloutInfoList rolloutInfoList =
(
	local iniFile = PxGetIniFilename()
	for rolloutInfoVal in rolloutInfoList collect  -- create a list...
	(
		local iniKeyName = ("RolloutRolledUp_" + rolloutInfoVal.rolloutVal.name)
		local iniVal = GetINISetting iniFile "RolloutInfo" iniKeyName
		local rolloutRolledUp = unsupplied  -- default if not stored in the ini file
		if (iniVal != "") then rolloutRolledUp = (iniVal as BooleanClass)  -- convert to bool if possible, or leave as unsupplied

		rolloutInfoVal.rolloutRolledUp = rolloutRolledUp
	)
)

-- Save state info from ini file, for each rollout in the list
fn PxSaveRolloutInfoList rolloutInfoList =
(
	PxUpdateRolloutInfoList rolloutInfoList
		
	local iniFile = PxGetIniFilename()
	for rolloutInfoVal in rolloutInfoList do
	(
		local rolloutRolledUp = rolloutInfoVal.rolloutRolledUp  -- fetch state information as updated above
		if (isKindOf rolloutRolledUp BooleanClass) then  -- proceed if value is boolean, skip if undefined or unsupplied
		(
			local iniKeyName = ("RolloutRolledUp_" + rolloutInfoVal.rolloutVal.name)			
			local iniVal = rolloutRolledUp as string
				
			SetINISetting iniFile "RolloutInfo" iniKeyName iniVal
		)
	)
)


---------- ---------- ---------- ----------
-- Multi-Editor Definitions
---------- ---------- ---------- ----------

struct MultiEdit_ControlInfo_struct  -- Subclass of KeyVal_struct
(
	keyVal = undefined,
	paramIdent = undefined, -- The parameter connected to the rollout control
	controlItem = undefined, -- The rollout control
	controlName = undefined, -- The rollout control UI name
	controlTooltip = undefined, -- The rollout control UI tooltip
	controlValLookup = undefined, -- The mapping from controlVal to paramVal.  Instance of ControlValMap_struct.  (optional)
	paramValArray = undefined -- This parameter value array, for list parameters
)

---------- ---------- ---------- ----------
-- 
---------- ---------- ---------- ----------

--
global gPxCreationHelper = undefined
global gPxCreationParent = undefined
global gPxCreationChild = undefined

tool PxCreationPlaceAndSizeTool
(
	on start do 
	(
		--hide gPxCreationHelper
	)
	
	on mousePoint clickno do
	(
		#stop	
	)

	on freeMove do
	(
		if gPxCreationHelper != undefined then
		(
			local helperPoint = gPxCreationHelper.pos * viewport.getTM()
			local viewCoord = mapScreenToView viewPoint helperPoint.z 
			gPxCreationHelper.helpersize = distance helperPoint viewCoord
		)
	)
)

-- add name support for different types of constraints. US3372
fn PxConstraintDefaultName type =
(
	case type of
	(
	gPxCreateConstraintRigidMode: "RigidConstraint"
	gPxCreateConstraintSlideMode: "SlideConstraint"
	gPxCreateConstraintHingeMode: "HingeConstraint"
	gPxCreateConstraintTwistMode: "TwistConstraint"
	gPxCreateConstraintUnivMode: "UniversalConstraint"
	gPxCreateConstraintBallSocketMode: "BallSocketConstraint"
	--gPxCreateConstraintGearMode: "GConstraint"             -- US3370, Remove constraint gears
	default: "Constraint"
	)
)

fn PxPrepareConstraintName type =
(
	subname = PxConstraintDefaultName(type)
	index = 1
	for i in objects do
	(
		if classof(i) == nvConstraint then
		(
			t = trimleft i.name subname
			c = t as integer
			if c != undefined and c >= index then index = c + 1
		)
	)
	if index < 10 then subname += "0"
	subname += (index as string)
)

fn PxInitJointParams joint type =
(
	joint.linearModeX = PX_DOF_LOCKED
	joint.linearModeY = PX_DOF_LOCKED
	joint.linearModeZ = PX_DOF_LOCKED
	joint.swing1Mode  = PX_DOF_LOCKED
	joint.swing2Mode  = PX_DOF_LOCKED
	joint.twistMode   = PX_DOF_LOCKED
		
	if type == gPxCreateConstraintSlideMode then 
	(
		joint.linearModeY = PX_DOF_LIMITED
	)
	else if type == gPxCreateConstraintHingeMode then (
		joint.swing1Mode = PX_DOF_LIMITED
		joint.swing1Angle = 100
	)
	else if type == gPxCreateConstraintTwistMode then (
		joint.twistMode = PX_DOF_FREE
	)
	else if type == gPxCreateConstraintUnivMode then (
		joint.swing1Mode = PX_DOF_LIMITED
		joint.swing2Mode = PX_DOF_LIMITED
	)
	else if type == gPxCreateConstraintBallSocketMode then (
		joint.swing1Mode = PX_DOF_LIMITED
		joint.swing1Angle = 80
		joint.swing2Mode = PX_DOF_LIMITED
		joint.swing2Angle = 80
		joint.twistMode  = PX_DOF_FREE
	)
	else if type == gPxCreateConstraintGearMode then (
		joint.gearing = true
		joint.linearModeX = PX_DOF_FREE
		joint.linearModeY = PX_DOF_FREE
		joint.linearModeZ = PX_DOF_FREE
		joint.swing1Mode = PX_DOF_FREE
		joint.swing2Mode = PX_DOF_FREE
		joint.twistMode  = PX_DOF_FREE
		joint.useProjection = true
		joint.projectionMode = gPxJointProjAngular
	)
	
	-- init joint names for different types
	joint.name = PxPrepareConstraintName(type)
)

fn PxCalculateConstraintPose np nc = 
(
	local t = nc.pos-np.pos;
	t.z = 0; 
	t = normalize t; 
	az = (acos t.x); 
	if t.y < 0 do az = -az; 
	
	d = (nc.pos-np.pos);
	r = length d;
	ay = -asin (d.z/r);

	--format "az = % zy = % t = %\n" az ay (nc.pos-np.pos)

	eu = eulerAngles 0 ay az; 
	q = -(eulerToQuat eu order:1); 
	tr = Matrix3 1; 
	tr.rotation = q; 
	tr.pos =np.pos; 
	tr
)

fn PxCreateConstraintForNodes parent child =
(
	local joint = undefined
	try(
		gPxCreationParent = parent
		gPxCreationChild = child
		
		bindToNode = parent
		if parent == undefined then bindToNode = child
		
		if bindToNode == undefined then undefined  -- result value
		else
		(		
			-- create constraint helper
			joint = nvConstraint()
			if joint == undefined then undefined  -- result value
			else
			(
				joint.Init()
		
				if parent == undefined or child == undefined then
					joint.pos = bindToNode.pos
				else if parent.pos == child.pos then
					joint.pos = bindToNode.pos
				else
					joint.transform = PxCalculateConstraintPose parent child

				joint.body0 = parent
				if child != undefined then joint.body1 = child
		
				joint.parent = parent
	
				PxInitJointParams joint gPxCreateConstraintTypeFlag
		
				-- drag and draw to resize the joint
				gPxCreationHelper = joint

				unhide gPxCreationHelper
				
				starttool PxCreationPlaceAndSizeTool prompt:nvpxText.globalsCreationToolPrompt

				joint = gPxCreationHelper
				gPxCreationHelper = undefined
			)
		)
	
	)catch()

	joint  -- result value
)

fn PxCreateConstraintAt pos type parent child = 
(
	joint = undefined
	if parent == undefined and child == undefined then undefined  -- result value
	else
	(
		if pos == undefined then pos = [0, 0, 0]
		if type == undefined then type = 1
		if type < gPxCreateConstraintRigidMode or type > gPxCreateConstraintGearMode then type = gPxCreateConstraintRigidMode
	
		-- force to make parent and child as RB
		modRB1 = PxGetModRB parent
		modRB2 = PxGetModRB child
		if parent == undefined and modRB2 == undefined then
		(
			modRB2 = PxMakeDynamicRB child
		)
		if child == undefined and modRB1 == undefined then
		(
			modRB1 = PxMakeDynamicRB parent
		)
	
		-- there is group node which can not be made as RB
		if modRB1 == undefined and modRB2 == undefined then undefined  -- result value
		else
		(
			-- make sure at least one dynamic RB
			if modRB1 == undefined then
			(
				if modRB2 == undefined then
				(
					format "Error: PxCreateConstraintAt(), impossible case.\n"
				)
				else
				(
					if modRB2.type != PX_PHYSTYPE_DYNAMIC then modRB2.type = PX_PHYSTYPE_DYNAMIC
				)
			)
			else
			(
				if modRB2 == undefined then
				(
					if modRB1.type != PX_PHYSTYPE_DYNAMIC then modRB1.type = PX_PHYSTYPE_DYNAMIC
				)
				else
				(
					if (modRB1.type != PX_PHYSTYPE_DYNAMIC) and (modRB2.type != PX_PHYSTYPE_DYNAMIC) then 
					(
						modRB1.type = PX_PHYSTYPE_DYNAMIC
					)
				)
			)

			joint = nvConstraint()
			joint.Init()
			joint.pos = pos
			joint.body0 = parent
			joint.body1 = child
			if parent != undefined then joint.parent = parent
			--joint.helpersize = 

			PxInitJointParams joint type

			joint  -- result value
		)
	)
)

fn PxCreateConstraint =
(
	isCancelled = false
	joint = undefined
	if (PxGetSelectionCount() == 1) then 
	(
		selectedNodes = PxGetSelectedNodes()
		child = selectedNodes[1]
		if not PxIsRB(child) then
		(
			local strStream = StringStream("")
			format nvpxText.constraintQueryApplyRBMTo1 child.name to:strStream
			info = nvpxText.constraintQueryMustBeRB + (strStream as string)
			yes  = queryBox info
			if yes then PxMakeDynamicRB child else isCancelled = true
		)
		else if PxIsStaticRB(child) then
		(
			messagebox nvpxText.constraintMsgStaticRBNotPartOfConstraint
			isCancelled = true
		)
		else if PxIsKinematicRB(child) then
		(
			messagebox nvpxText.constraintMsgNoOnSingleKRB
			isCancelled = true
		)
		if not isCancelled do
		(
			joint = PxCreateConstraintForNodes undefined child
		)
	)
	else if (PxGetSelectionCount() == 2) then
	(
		selectedNodesArray = PxGetSelectedNodes()
		parent = selectedNodesArray[1]
		child  = selectedNodesArray[2]
		if not PxIsRB(parent) then
		(
			if not PxIsRB(child) then
			(
				local strStream = StringStream("")
				format nvpxText.constraintQueryApplyRBMTo2 parent.name child.name to:strStream
				info = nvpxText.constraintQueryMustBeRB + (strStream as string)
				yes  = queryBox info
				if yes then (PxMakeDynamicRB parent; PxMakeDynamicRB child) else isCancelled = true
			)
			else
			(
				local strStream = StringStream("")
				format nvpxText.constraintQueryApplyRBMTo1 parent.name to:strStream
				info = nvpxText.constraintQueryMustBeRB + (strStream as string)
				yes  = queryBox info
				if yes then PxMakeDynamicRB parent else isCancelled = true
			)
		)
		else if not PxIsRB(child) then
		(
			local strStream = StringStream("")
			format nvpxText.constraintQueryApplyRBMTo1 child.name to:strStream
			info = nvpxText.constraintQueryMustBeRB + (strStream as string)
			yes  = queryBox info
			if yes then PxMakeDynamicRB child else isCancelled = true
		)
		
		if not isCancelled do
		(

			if PxIsStaticRB(parent) or PxIsStaticRB(child) then
			(
				messagebox nvpxText.constraintMsgStaticRBNotPartOfConstraint
				isCancelled = true
			)
			if PxIsKinematicRB(parent) and PxIsKinematicRB(child) then
			(
				messagebox nvpxText.constraintMsgConstraintWith2KRB
				isCancelled = true
			)
		)

		if not isCancelled do
		(
			joint = PxCreateConstraintForNodes parent child
		)
	)
	if joint != undefined then
	(
		PxInitJointParams joint gPxCreateConstraintTypeFlag

		select joint
		max modify mode
	)

	joint  -- result value
)

fn PxCreateConstrintRigid =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintRigidMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintSlide =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintSlideMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintHinge =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintHingeMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintTwist =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintTwistMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintUniv =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintUnivMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintBallSocket =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintBallSocketMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxCreateConstrintGear =
(
	gPxCreateConstraintTypeFlag = gPxCreateConstraintGearMode

	if gPxCreationHelper != undefined then false  -- result value
	else
	(
		joint = PxCreateConstraint()
		gPxCreateConstraintTypeFlag = 0
	)
)

fn PxNodeIsConvex n =
(
	if not nvpx.IsSimulating() do nvpx.RemoveAll()
	nvpx.isConvex(n)
)

fn PxGetRagdollRootBone n =
(
	root = PxGetRootParent(n)
	handle = (getuserprop root "RagdollMesh")
	if handle != undefined then (
		root
	) else if PxIsBone(n) then (
		root
	)
	else (
		root = nvpx.RagdollGetRootBone n
	)
)

fn PxPotentialRagdoll n =
(
	local retval = undefined
	if PxIsBone(n) then retval = true
	else
	(
		m = n.modifiers[#Physique]
		if m == undefined do m = n.modifiers[#Skin]
		root = PxGetRootParent(n)
		if root != undefined do (
			handle = (getuserprop root "RagdollMesh")
			if handle != undefined then retval = true
		)
		if retval==undefined do retval (m != undefined)
	)
	retval  -- result value
)


fn PxGetOldRagdoll n =
(
	local retval = undefined
	if classof(n) == nvRagdoll then retval = n
	else
	(
		root = PxGetRootParent(n)
		for i in objects while retval==undefined do (
			if classof(i) == nvRagdoll then (
				--format " root = %, bone = %, rag = %\n" root i.rootBone i.ragdollNode
				if i.rootBone == root do retval = i
				if i.ragdollNode == root do retval = i
			)
		)
	)
	retval
)

fn PxRemoveOldRagdoll =
(
	currentNode = PxGetCurrentNode()
	if currentNode !=undefined then (
		rag = PxGetOldRagdoll(currentNode)
		if rag != undefined then (
			info = nvpxText.globalsRemoveOldRagdoll
			yes  = queryBox info
			if yes then (
				rag.initBoneArrays()
				rag.deleteJoints()
				rag.deleteRBs()
			)
			delete rag
		)
	)
)

fn PxRemovePhysXMods node =
(
	if (classof(node)==Dummy and superclassof(node)==Helper) then (
		children = #()
		PxGetAllChildrenNodes node children
		for j in children do (
			PxDeleteModRB j
		)
	) 
	PxDeleteRB node
)

global gPxCreatingCount = 0
fn PxStopSecond n =
(
	--format "creating function\n"
	gPxCreatingCount = gPxCreatingCount + 1
	if gPxCreatingCount > 1 then (
		StopCreating()
		gPxCreatingCount = 0
	)
)


fn PxCreateBox transform length width height =
(
	c = Box()
	c.transform = transform
	c.length = length
	c.width = width
	c.height = height
	c
)

fn PxCreateSphere transform radius =
(
	c = Sphere()
	c.transform = transform
	c.radius = radius
	c
)


fn PxCreateCapsule transform height radius =
(
	c = capsule()
	c.transform = transform
	c.heightType = 1  -- only center part
	c.height = height
	c.radius = radius
	c
)

-- called in Apex Clothing -> "Import Clothing Template..."
fn PxImportTemplate = 
(
	-- creating dialog filename mask
	local format_ext = "ctw";
	local format_name = "CTW";
	local filename = "";
	local searchmask = format_name + " (*." + format_ext + ")|*." + format_ext;

	-- getting the filename of the file to import
	filename = getopenfilename filename:filename types:searchmask caption:nvpxText.globalsImportTemplateCaption

	if ( filename != undefined ) then
	(
		nvpx.LoadClothingTemplate filename
	)
	else
	(
		format nvpxText.TXT_GLOBALS_IMPORT_ERR_FMT filename
	)
)

-- called in Apex Clothing -> "Export Clothing Template..."
fn PxExportTemplate = 
(
	-- creating dialog filename mask
	local format_ext = "ctw";
	local format_name = "CTW";
	local fullfilename = "";
	local searchmask = format_name + " (*." + format_ext + ")|*." + format_ext;

	-- getting the filename of the file to export
	fullfilename = getsavefilename filename:fullfilename types:searchmask caption:nvpxText.globalsExportTemplateCaption

	if (fullfilename != undefined) then
	(
		local filename = getFilenameFile fullfilename
		local filepath = getFilenamePath fullfilename
		nvpx.SaveClothingTemplate filepath filename
	)
	else
	(
		format nvpxText.TXT_GLOBALS_EXPORT_ERR_FMT filename
	)
)

-- called in Apex Clothing -> "Rebuild Selected Clothing"
fn PxRebuildSelectedClothing = 
(
	nvpx.RebuildSelectedClothingNode()
)

global gPxSimClockLastTime	= 0
global PX_LASTFRAME_CONTINUE= nvpxConsts.PX_ONLASTFRAME_CONTINUE
global PX_LASTFRAME_PAUSE	= nvpxConsts.PX_ONLASTFRAME_STOP
global PX_LASTFRAME_LOOP	= nvpxConsts.PX_ONLASTFRAME_LOOP

global PX_SIMLOOP_RESTART	= nvpxConsts.PX_LOOP_RESET
global PX_SIMLOOP_REPEAT	= nvpxConsts.PX_LOOP_CONTINUE

fn PxSimClock =
(
		global gPxSimClockLastTime
		tDelta = 1000.0/(frameRate); 
		
		--DETERMINE BURN TIME IF SIMULATING TOO FAST--		
		tStart = timeStamp()
		tEnd = gPxSimClockLastTime + tDelta
		if (tEnd > tStart) then
		(
			while timeStamp() < tEnd do
			(
			)
		)
		gPxSimClockLastTime = tEnd
		if(gPxSimClockLastTime < tStart) then gPxSimClockLastTime = tStart
		
		with redraw off 
		(
			undo off
			(
				try
				(
					PxSimulateOneFrame(); 
				) catch ()
			)
		)
		
		redrawviews();
		
		if (sliderTime >= animationRange.end) then
		(	
			case gPxOnLastFrame of
			(
				PX_LASTFRAME_CONTINUE:  slidertime = animationRange.end
				PX_LASTFRAME_PAUSE: PxPauseSimulation()
				PX_LASTFRAME_LOOP:  if gPxLoopAnimation == PX_SIMLOOP_RESTART then (PxStopSimulation(); PxRunSimulation()) else sliderTime = animationRange.start
			)
		)	
)

global gSimClock = dotNetObject "System.Windows.Forms.Timer" 
dotnet.addEventHandler  gSimClock "tick" PxSimClock
gSimClock.interval = 1

fn PxGetSimPlaybackSetting =
(
	ret = gPxOnLastFrame
	if gPxOnLastFrame == PX_LASTFRAME_LOOP do
	(
		--if gPxLoopAnimation == PX_SIMLOOP_RESTART then ret = 3 else ret = 4
		ret = gPxLoopAnimation + 2
	)
	ret
)

fn PxSetSimPlaybackSetting v = 
(
	case v of
	(
	PX_LASTFRAME_CONTINUE:  (gPxOnLastFrame = PX_LASTFRAME_CONTINUE;  )
	PX_LASTFRAME_PAUSE: (gPxOnLastFrame = PX_LASTFRAME_PAUSE; )
	PX_LASTFRAME_LOOP:  (gPxOnLastFrame = PX_LASTFRAME_LOOP; gPxLoopAnimation = PX_SIMLOOP_RESTART)
	PX_LASTFRAME_LOOP:  (gPxOnLastFrame = PX_LASTFRAME_LOOP; gPxLoopAnimation = PX_SIMLOOP_REPEAT)
	)
	
	global physXpaneldata
	physXpaneldata.onLastFrame	 = gPxOnLastFrame
	physXpaneldata.loopAnimation = gPxLoopAnimation
	
	if px_panel_simulation.open == true then
	(
		px_panel_simulation.loadValues()
	)
)

fn PxCheckSDK3 =
(
	PxSDK3Loaded = false
	for i = 1 to nvpx.GetPhysXPluginCount() do
	(
		if nvpx.GetPhysXPluginVersionMajor(i) == 3 do PxSDK3Loaded = true
	)
	PxSDK3Loaded
)

-- run to check whether SDK3 avaliable or not
PxCheckSDK3()

fn PxSwitchSDK verMajor silent =
(
	local targetVersion = "";
	local switchOk = false
	for i = 1 to nvpx.GetPhysXPluginCount() do
	(
		if nvpx.GetPhysXPluginVersionMajor(i) == verMajor do
		(
			targetVersion = nvpx.GetPhysXPluginVersionString(i)  -- get target SDK version string
			clearSelection()
			switchOk = nvpx.usePhysXPlugin(i);
			exit
		)
	)
	
	local sdkVersion = nvpx.GetPhysXSDKVersionString()

	if switchOk then
	(
		if not silent do MessageBox ("Switched! Now you are using PhysX " + sdkVersion + ".") title:"PhysX SDK"
	)
	else
	(
		if not silent do MessageBox ("PhysX " + targetVersion + " is not loaded. You are still using PhysX " + sdkVersion + ".") title:"PhysX SDK"
	)
	switchOk
)

fn PxSwitchSDKSilent verMajor =
(
	PxSwitchSDK verMajor true
)

fn hasMacroRecorderContext =
(
	(MaxVersion())[1] > PX_MAXVERSION_2012
)

fn TurnOnMacroRecorderContext onOff =
(
	if (hasMacroRecorderContext()) do (
		-- 'set macroRecorderEmitterEnabled on' returns the previous value of the recorder disable count, 
		-- and decrements it (disables it less)
		if (onOff) then
			execute "set macroRecorderEmitterEnabled on"
		else 
			execute "set macroRecorderEmitterEnabled off"
	)
)

fn PxSetProperty itemVal propID propVal =
(	-- Enables the property change to be macro-recorded.  No effect unless macro-recording is enabled by the user.
	TurnOnMacroRecorderContext true
	setProperty itemVal propID propVal
	TurnOnMacroRecorderContext false
)



-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQC3Z75vlV3c+aDJeN1M77p11JoUUFBMDast
-- b6/tWTHRSL+rm23k7/gnyNLPX03uj8L1yVTFDZTCxK2979B3wBaJUVpimLwVmLh8
-- sSx0mqyqi3q0NXpGQroPHnYogRzTChzp3RSP7yOC3IbGKH/cEq7ptRQHR1NNZnQ2
-- mEPk9Nw0cASNFTjYAJBKLaTWn/zGBVtGPiwJ6l9NHLD1daHpxeW+T8oei48wBFoS
-- MQ+S45SrAcrG6eT1wGRencJvhVmbQ8l0xA9yUeK+xo1PaG8PvuxIpRwVhZKJWTAC
-- i7vAqyIgpcZQqfcuRORn+W5W967jT1HGbnE2VjU521BEPEYOvg5E
-- -----END-SIGNATURE-----